const els = {
  loginCard: document.getElementById("login-card"),
  dashboard: document.getElementById("dashboard"),
  loginUsername: document.getElementById("login-username"),
  loginPassword: document.getElementById("login-password"),
  loginBtn: document.getElementById("login-btn"),
  refreshBtn: document.getElementById("refresh-btn"),
  logoutBtn: document.getElementById("logout-btn"),
  status: document.getElementById("status"),
  statTotalSessions: document.getElementById("stat-total-sessions"),
  statActiveSessions: document.getElementById("stat-active-sessions"),
  statTotalEvents: document.getElementById("stat-total-events"),
  statSessionsToday: document.getElementById("stat-sessions-today"),
  sessionsBody: document.getElementById("sessions-body"),
  logsBody: document.getElementById("logs-body"),
  settingSiteName: document.getElementById("setting-site-name"),
  settingNotice: document.getElementById("setting-notice"),
  settingLimit: document.getElementById("setting-limit"),
  saveSettingsBtn: document.getElementById("save-settings-btn"),
  newDomainInput: document.getElementById("new-domain-input"),
  addDomainBtn: document.getElementById("add-domain-btn"),
  domainsBody: document.getElementById("domains-body"),
  changeCurrentPassword: document.getElementById("change-current-password"),
  changeNewPassword: document.getElementById("change-new-password"),
  changePasswordBtn: document.getElementById("change-password-btn")
};

function setStatus(text, isError = false) {
  if (!els.status) return;
  els.status.textContent = text;
  els.status.style.color = isError ? "#fecaca" : "#a7f3d0";
}

async function api(path, options = {}) {
  const response = await fetch(path, {
    headers: {
      "Content-Type": "application/json"
    },
    ...options
  });
  const payload = await response.json().catch(() => ({}));
  if (!response.ok) {
    throw new Error(payload.error || "Request failed");
  }
  return payload;
}

function showDashboard() {
  els.loginCard.classList.add("hidden");
  els.dashboard.classList.remove("hidden");
}

function showLogin() {
  els.dashboard.classList.add("hidden");
  els.loginCard.classList.remove("hidden");
}

async function checkAuth() {
  try {
    await api("/api/admin/me");
    showDashboard();
    await loadAllData();
  } catch (_error) {
    showLogin();
  }
}

async function login() {
  const username = els.loginUsername.value.trim();
  const password = els.loginPassword.value;
  await api("/api/admin/login", {
    method: "POST",
    body: JSON.stringify({ username, password })
  });
  showDashboard();
  setStatus("Logged in.");
  await loadAllData();
}

async function logout() {
  await api("/api/admin/logout", { method: "POST" });
  showLogin();
}

function renderStats(stats) {
  els.statTotalSessions.textContent = stats.total_sessions ?? 0;
  els.statActiveSessions.textContent = stats.active_sessions ?? 0;
  els.statTotalEvents.textContent = stats.total_events ?? 0;
  els.statSessionsToday.textContent = stats.sessions_today ?? 0;
}

function renderSessions(items) {
  els.sessionsBody.innerHTML = "";
  for (const session of items) {
    const tr = document.createElement("tr");
    const statusClass = session.is_active ? "active" : "blocked";
    const statusText = session.is_active ? "Active" : "Disabled";

    tr.innerHTML = `
      <td>${session.id}</td>
      <td>${escapeHtml(session.email_addr)}</td>
      <td><span class="status-pill ${statusClass}">${statusText}</span></td>
      <td>${escapeHtml(session.last_ip || "-")}</td>
      <td>${escapeHtml(session.updated_at || "-")}</td>
      <td>
        <button data-id="${session.id}" data-next="${session.is_active ? 0 : 1}">
          ${session.is_active ? "Disable" : "Enable"}
        </button>
      </td>
    `;
    els.sessionsBody.appendChild(tr);
  }
}

function renderLogs(logs) {
  els.logsBody.innerHTML = "";
  for (const log of logs) {
    const tr = document.createElement("tr");
    tr.innerHTML = `
      <td>${log.id}</td>
      <td>${log.session_id}</td>
      <td>${escapeHtml(log.email_addr || "-")}</td>
      <td>${escapeHtml(log.event_type)}</td>
      <td><code>${escapeHtml(JSON.stringify(log.payload || {}))}</code></td>
      <td>${escapeHtml(log.created_at || "-")}</td>
    `;
    els.logsBody.appendChild(tr);
  }
}

function renderDomains(domains) {
  els.domainsBody.innerHTML = "";
  if (!domains.length) {
    const tr = document.createElement("tr");
    tr.innerHTML = `<td colspan="2">No domains added yet.</td>`;
    els.domainsBody.appendChild(tr);
    return;
  }

  for (const domain of domains) {
    const tr = document.createElement("tr");
    tr.innerHTML = `
      <td>${escapeHtml(domain)}</td>
      <td>
        <button data-domain="${escapeHtml(domain)}">Delete</button>
      </td>
    `;
    els.domainsBody.appendChild(tr);
  }
}

function renderSettings(settings) {
  els.settingSiteName.value = settings.site_name || "";
  els.settingNotice.value = settings.notice || "";
  els.settingLimit.value = Number(settings.messages_limit || 50);
}

async function loadAllData() {
  setStatus("Loading dashboard data...");
  const [stats, sessions, logs, settings, domains] = await Promise.all([
    api("/api/admin/stats"),
    api("/api/admin/sessions?limit=100"),
    api("/api/admin/logs?limit=120"),
    api("/api/admin/settings"),
    api("/api/admin/domains")
  ]);
  renderStats(stats);
  renderSessions(sessions.sessions || []);
  renderLogs(logs.logs || []);
  renderSettings(settings.settings || {});
  renderDomains(domains.domains || []);
  setStatus("Dashboard synced.");
}

async function saveSettings() {
  const messagesLimit = Math.max(1, Math.min(200, Number(els.settingLimit.value || 50)));

  await api("/api/admin/settings", {
    method: "PUT",
    body: JSON.stringify({
      settings: {
        site_name: els.settingSiteName.value,
        notice: els.settingNotice.value,
        messages_limit: String(messagesLimit)
      }
    })
  });
  setStatus("Settings saved.");
}

async function addDomain() {
  const domain = els.newDomainInput.value.trim();
  await api("/api/admin/domains", {
    method: "POST",
    body: JSON.stringify({ domain })
  });
  els.newDomainInput.value = "";
  setStatus("Domain added.");
}

async function deleteDomain(domain) {
  await api(`/api/admin/domains/${encodeURIComponent(domain)}`, {
    method: "DELETE"
  });
  setStatus("Domain deleted.");
}

async function changePassword() {
  await api("/api/admin/change-password", {
    method: "POST",
    body: JSON.stringify({
      currentPassword: els.changeCurrentPassword.value,
      newPassword: els.changeNewPassword.value
    })
  });
  els.changeCurrentPassword.value = "";
  els.changeNewPassword.value = "";
  setStatus("Password updated.");
}

function escapeHtml(str) {
  return String(str)
    .replaceAll("&", "&amp;")
    .replaceAll("<", "&lt;")
    .replaceAll(">", "&gt;")
    .replaceAll('"', "&quot;")
    .replaceAll("'", "&#039;");
}

function bindEvents() {
  els.loginBtn.addEventListener("click", async () => {
    try {
      await login();
    } catch (error) {
      setStatus(error.message, true);
    }
  });

  els.refreshBtn.addEventListener("click", async () => {
    try {
      await loadAllData();
    } catch (error) {
      setStatus(error.message, true);
    }
  });

  els.logoutBtn.addEventListener("click", async () => {
    try {
      await logout();
      setStatus("Logged out.");
    } catch (error) {
      setStatus(error.message, true);
    }
  });

  els.sessionsBody.addEventListener("click", async (event) => {
    const button = event.target.closest("button[data-id]");
    if (!button) return;
    try {
      await api(`/api/admin/sessions/${button.dataset.id}/status`, {
        method: "PATCH",
        body: JSON.stringify({ isActive: button.dataset.next === "1" })
      });
      await loadAllData();
    } catch (error) {
      setStatus(error.message, true);
    }
  });

  els.saveSettingsBtn.addEventListener("click", async () => {
    try {
      await saveSettings();
      await loadAllData();
    } catch (error) {
      setStatus(error.message, true);
    }
  });

  els.addDomainBtn.addEventListener("click", async () => {
    try {
      await addDomain();
      await loadAllData();
    } catch (error) {
      setStatus(error.message, true);
    }
  });

  els.domainsBody.addEventListener("click", async (event) => {
    const button = event.target.closest("button[data-domain]");
    if (!button) return;
    try {
      await deleteDomain(button.dataset.domain);
      await loadAllData();
    } catch (error) {
      setStatus(error.message, true);
    }
  });

  els.changePasswordBtn.addEventListener("click", async () => {
    try {
      await changePassword();
    } catch (error) {
      setStatus(error.message, true);
    }
  });
}

bindEvents();
checkAuth();
