const state = {
  sessionToken: localStorage.getItem("temp_mail_session_token") || "",
  emailAddress: "",
  autoRefreshTimer: null,
  hasDomains: false
};

const els = {
  siteName: document.getElementById("site-name"),
  siteNotice: document.getElementById("site-notice"),
  domainSelect: document.getElementById("domain-select"),
  mailbox: document.getElementById("mailbox"),
  status: document.getElementById("status"),
  messagesBody: document.getElementById("messages-body"),
  newBtn: document.getElementById("new-btn"),
  regenBtn: document.getElementById("regen-btn"),
  copyBtn: document.getElementById("copy-btn"),
  refreshBtn: document.getElementById("refresh-btn"),
  deleteAllBtn: document.getElementById("delete-all-btn"),
  autoRefresh: document.getElementById("auto-refresh"),
  modal: document.getElementById("message-modal"),
  modalClose: document.getElementById("modal-close"),
  modalSubject: document.getElementById("modal-subject"),
  modalFrom: document.getElementById("modal-from"),
  modalDate: document.getElementById("modal-date"),
  modalContent: document.getElementById("modal-content")
};

function setStatus(text, isError = false) {
  els.status.textContent = text;
  els.status.style.color = isError ? "#b91c1c" : "#475569";
}

async function api(path, options = {}) {
  const response = await fetch(path, {
    headers: {
      "Content-Type": "application/json"
    },
    ...options
  });

  const payload = await response.json().catch(() => ({}));
  if (!response.ok) {
    throw new Error(payload.error || "Request failed");
  }
  return payload;
}

function saveToken(token) {
  state.sessionToken = token;
  localStorage.setItem("temp_mail_session_token", token);
}

function renderMessageRows(messages) {
  els.messagesBody.innerHTML = "";

  if (!messages.length) {
    const row = document.createElement("tr");
    row.innerHTML = `<td colspan="5">No messages yet.</td>`;
    els.messagesBody.appendChild(row);
    return;
  }

  for (const msg of messages) {
    const tr = document.createElement("tr");
    const date = msg.mail_timestamp
      ? new Date(msg.mail_timestamp * 1000).toLocaleString()
      : "-";

    tr.innerHTML = `
      <td>${msg.mail_id}</td>
      <td>${escapeHtml(msg.mail_from || "-")}</td>
      <td>${escapeHtml(msg.mail_subject || "(No Subject)")}</td>
      <td>${escapeHtml(date)}</td>
      <td>
        <button class="view-btn" data-id="${msg.mail_id}">View</button>
        <button class="delete-btn" data-id="${msg.mail_id}">Delete</button>
      </td>
    `;
    els.messagesBody.appendChild(tr);
  }
}

function escapeHtml(str) {
  return String(str)
    .replaceAll("&", "&amp;")
    .replaceAll("<", "&lt;")
    .replaceAll(">", "&gt;")
    .replaceAll('"', "&quot;")
    .replaceAll("'", "&#039;");
}

async function loadOptions() {
  const options = await api("/api/public/options");
  els.siteName.textContent = options.siteName || "Temp Mail Control";
  els.siteNotice.textContent = options.notice || "";
  const domains = Array.isArray(options.domains) ? options.domains : [];

  if (!domains.length) {
    state.hasDomains = false;
    els.domainSelect.innerHTML = `<option value="">No domain configured</option>`;
    els.domainSelect.disabled = true;
    els.newBtn.disabled = true;
    els.regenBtn.disabled = true;
    setStatus("Admin must add at least one domain from admin panel.", true);
    return;
  }

  state.hasDomains = true;
  els.domainSelect.disabled = false;
  els.newBtn.disabled = false;
  els.regenBtn.disabled = false;
  els.domainSelect.innerHTML = domains
    .map((domain) => `<option value="${domain}">${domain}</option>`)
    .join("");
}

async function createNewInbox() {
  if (!state.hasDomains) {
    setStatus("No domain configured by admin.", true);
    return;
  }
  const domain = els.domainSelect.value;
  setStatus("Creating mailbox...");
  const result = await api("/api/public/new", {
    method: "POST",
    body: JSON.stringify({ domain })
  });
  saveToken(result.sessionToken);
  state.emailAddress = result.emailAddress;
  els.mailbox.value = result.emailAddress;
  setStatus("Mailbox created.");
  await refreshMessages();
}

async function regenerateInbox() {
  if (!state.hasDomains) {
    setStatus("No domain configured by admin.", true);
    return;
  }
  if (!state.sessionToken) {
    await createNewInbox();
    return;
  }
  setStatus("Regenerating mailbox...");
  const result = await api("/api/public/regenerate", {
    method: "POST",
    body: JSON.stringify({
      sessionToken: state.sessionToken,
      domain: els.domainSelect.value
    })
  });
  state.emailAddress = result.emailAddress;
  els.mailbox.value = result.emailAddress;
  setStatus("Address regenerated.");
  await refreshMessages();
}

async function loadExistingSession() {
  if (!state.sessionToken) {
    await createNewInbox();
    return;
  }
  try {
    const session = await api(
      `/api/public/session?sessionToken=${encodeURIComponent(state.sessionToken)}`
    );
    state.emailAddress = session.emailAddress;
    els.mailbox.value = session.emailAddress;
    await refreshMessages();
  } catch (_error) {
    await createNewInbox();
  }
}

async function refreshMessages() {
  if (!state.sessionToken) {
    return;
  }
  setStatus("Loading inbox...");
  const data = await api(
    `/api/public/messages?sessionToken=${encodeURIComponent(state.sessionToken)}`
  );
  renderMessageRows(data.messages || []);
  setStatus(`Inbox synced. Total messages: ${data.count ?? 0}`);
}

async function viewMessage(messageId) {
  const data = await api(
    `/api/public/messages/${encodeURIComponent(
      messageId
    )}?sessionToken=${encodeURIComponent(state.sessionToken)}`
  );

  els.modalSubject.textContent = data.mail_subject || "(No Subject)";
  els.modalFrom.textContent = data.mail_from || "-";
  els.modalDate.textContent = data.mail_timestamp
    ? new Date(data.mail_timestamp * 1000).toLocaleString()
    : "-";

  if (data.mail_body) {
    els.modalContent.innerHTML = sanitizeHtml(data.mail_body);
  } else if (data.mail_excerpt) {
    els.modalContent.textContent = data.mail_excerpt;
  } else {
    els.modalContent.textContent = "No content available.";
  }

  els.modal.showModal();
}

function sanitizeHtml(unsafeHtml) {
  const parser = new DOMParser();
  const doc = parser.parseFromString(unsafeHtml, "text/html");
  const bannedTags = ["script", "iframe", "object", "embed", "form"];
  for (const tag of bannedTags) {
    doc.querySelectorAll(tag).forEach((el) => el.remove());
  }
  doc.querySelectorAll("*").forEach((node) => {
    for (const attr of [...node.attributes]) {
      const attrName = attr.name.toLowerCase();
      const attrValue = attr.value.toLowerCase();
      if (attrName.startsWith("on") || attrValue.startsWith("javascript:")) {
        node.removeAttribute(attr.name);
      }
    }
  });
  return doc.body.innerHTML;
}

async function deleteMessage(messageId) {
  await api(
    `/api/public/messages/${encodeURIComponent(
      messageId
    )}?sessionToken=${encodeURIComponent(state.sessionToken)}`,
    { method: "DELETE" }
  );
  await refreshMessages();
}

async function deleteAllMessages() {
  await api("/api/public/messages/delete-all", {
    method: "POST",
    body: JSON.stringify({ sessionToken: state.sessionToken })
  });
  await refreshMessages();
}

function setupEvents() {
  els.newBtn.addEventListener("click", async () => {
    try {
      await createNewInbox();
    } catch (error) {
      setStatus(error.message, true);
    }
  });

  els.regenBtn.addEventListener("click", async () => {
    try {
      await regenerateInbox();
    } catch (error) {
      setStatus(error.message, true);
    }
  });

  els.copyBtn.addEventListener("click", async () => {
    if (!state.emailAddress) return;
    try {
      await navigator.clipboard.writeText(state.emailAddress);
      setStatus("Email copied to clipboard.");
    } catch (_error) {
      setStatus("Clipboard access denied.", true);
    }
  });

  els.refreshBtn.addEventListener("click", async () => {
    try {
      await refreshMessages();
    } catch (error) {
      setStatus(error.message, true);
    }
  });

  els.deleteAllBtn.addEventListener("click", async () => {
    if (!state.sessionToken) return;
    if (!confirm("Delete all messages from this mailbox?")) return;
    try {
      await deleteAllMessages();
    } catch (error) {
      setStatus(error.message, true);
    }
  });

  els.messagesBody.addEventListener("click", async (event) => {
    const button = event.target.closest("button");
    if (!button) return;
    const id = button.getAttribute("data-id");
    if (!id) return;

    try {
      if (button.classList.contains("view-btn")) {
        await viewMessage(id);
      } else if (button.classList.contains("delete-btn")) {
        await deleteMessage(id);
      }
    } catch (error) {
      setStatus(error.message, true);
    }
  });

  els.modalClose.addEventListener("click", () => els.modal.close());

  els.autoRefresh.addEventListener("change", () => {
    const seconds = Number(els.autoRefresh.value || "0");
    if (state.autoRefreshTimer) {
      clearInterval(state.autoRefreshTimer);
      state.autoRefreshTimer = null;
    }
    if (seconds > 0) {
      state.autoRefreshTimer = setInterval(() => {
        refreshMessages().catch((error) => setStatus(error.message, true));
      }, seconds * 1000);
      setStatus(`Auto refresh enabled (${seconds}s).`);
    } else {
      setStatus("Auto refresh disabled.");
    }
  });
}

async function start() {
  setupEvents();
  try {
    await loadOptions();
    await loadExistingSession();
  } catch (error) {
    setStatus(error.message || "Failed to start app", true);
  }
}

start();
