const fs = require("fs");
const path = require("path");
const bcrypt = require("bcryptjs");
const sqlite3 = require("sqlite3");
const { open } = require("sqlite");

let dbPromise;

const LEGACY_DEFAULT_DOMAINS = [
  "guerrillamail.com",
  "guerrillamail.net",
  "guerrillamail.org",
  "guerrillamailblock.com",
  "grr.la",
  "pokemail.net",
  "spam4.me"
];

async function getDb() {
  if (!dbPromise) {
    const dataDir = path.join(process.cwd(), "data");
    fs.mkdirSync(dataDir, { recursive: true });

    dbPromise = open({
      filename: path.join(dataDir, "app.db"),
      driver: sqlite3.Database
    });
  }
  return dbPromise;
}

async function setDefaultSettingIfMissing(db, key, value) {
  await db.run(
    `INSERT OR IGNORE INTO site_settings (key, value, updated_at)
     VALUES (?, ?, CURRENT_TIMESTAMP)`,
    [key, value]
  );
}

async function ensureDefaultSettings(db) {
  await setDefaultSettingIfMissing(db, "site_name", "Temp Mail Control");
  await setDefaultSettingIfMissing(
    db,
    "notice",
    "Use temporary email responsibly."
  );
  await setDefaultSettingIfMissing(db, "domain_list", "[]");
  await setDefaultSettingIfMissing(db, "messages_limit", "50");

  // One-time cleanup: remove legacy prefilled domains if they still exist.
  const legacyJson = JSON.stringify(LEGACY_DEFAULT_DOMAINS);
  await db.run(
    `UPDATE site_settings
     SET value = '[]', updated_at = CURRENT_TIMESTAMP
     WHERE key = 'domain_list' AND value = ?`,
    [legacyJson]
  );
}

async function ensureDefaultAdmin(db) {
  const username = process.env.ADMIN_USERNAME || "admin";
  const password = process.env.ADMIN_PASSWORD || "ChangeMe123!";

  const existing = await db.get("SELECT id FROM admin_users WHERE username = ?", [
    username
  ]);

  if (!existing) {
    const hash = await bcrypt.hash(password, 12);
    await db.run(
      "INSERT INTO admin_users (username, password_hash) VALUES (?, ?)",
      [username, hash]
    );
    console.log(
      `Created default admin user "${username}". Change password after first login.`
    );
  }
}

async function initDb() {
  const db = await getDb();
  await db.exec("PRAGMA foreign_keys = ON");
  await db.exec(`
    CREATE TABLE IF NOT EXISTS admin_users (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      username TEXT NOT NULL UNIQUE,
      password_hash TEXT NOT NULL,
      created_at TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP,
      updated_at TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP
    );

    CREATE TABLE IF NOT EXISTS site_settings (
      key TEXT PRIMARY KEY,
      value TEXT NOT NULL,
      updated_at TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP
    );

    CREATE TABLE IF NOT EXISTS mail_sessions (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      public_token TEXT NOT NULL UNIQUE,
      sid_token TEXT NOT NULL,
      email_addr TEXT NOT NULL,
      is_active INTEGER NOT NULL DEFAULT 1,
      last_ip TEXT,
      created_at TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP,
      updated_at TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP
    );

    CREATE TABLE IF NOT EXISTS session_events (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      session_id INTEGER NOT NULL,
      event_type TEXT NOT NULL,
      payload TEXT,
      created_at TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY(session_id) REFERENCES mail_sessions(id) ON DELETE CASCADE
    );
  `);

  await ensureDefaultSettings(db);
  await ensureDefaultAdmin(db);
}

module.exports = {
  getDb,
  initDb
};
