const jwt = require("jsonwebtoken");

const COOKIE_NAME = "admin_token";

function getJwtSecret() {
  return process.env.JWT_SECRET || "dev-only-secret-change-this";
}

function signAdminToken(payload) {
  return jwt.sign(payload, getJwtSecret(), {
    expiresIn: process.env.JWT_EXPIRES_IN || "12h"
  });
}

function verifyToken(token) {
  return jwt.verify(token, getJwtSecret());
}

function requireAdmin(req, res, next) {
  const token = req.cookies[COOKIE_NAME];
  if (!token) {
    return res.status(401).json({ error: "Unauthorized" });
  }

  try {
    req.admin = verifyToken(token);
    return next();
  } catch (_error) {
    return res.status(401).json({ error: "Invalid or expired token" });
  }
}

function setAdminCookie(res, token) {
  const isProduction = process.env.NODE_ENV === "production";
  res.cookie(COOKIE_NAME, token, {
    httpOnly: true,
    secure: isProduction,
    sameSite: "lax",
    maxAge: 1000 * 60 * 60 * 12
  });
}

function clearAdminCookie(res) {
  res.clearCookie(COOKIE_NAME, { sameSite: "lax" });
}

module.exports = {
  COOKIE_NAME,
  signAdminToken,
  requireAdmin,
  setAdminCookie,
  clearAdminCookie
};
