const bcrypt = require("bcryptjs");
const express = require("express");
const { getDb } = require("../db");
const {
  signAdminToken,
  requireAdmin,
  setAdminCookie,
  clearAdminCookie
} = require("../middleware/auth");

const router = express.Router();

const SETTINGS_ALLOWLIST = new Set(["site_name", "notice", "messages_limit"]);

function safeJsonParse(value, fallback = null) {
  try {
    return JSON.parse(value);
  } catch (_error) {
    return fallback;
  }
}

function normalizeDomain(value) {
  return String(value || "").trim().toLowerCase();
}

function isValidDomain(value) {
  return /^[a-z0-9.-]+\.[a-z]{2,}$/i.test(value);
}

async function getDomainList(db) {
  const row = await db.get("SELECT value FROM site_settings WHERE key = 'domain_list'");
  const list = safeJsonParse(row?.value, []);
  if (!Array.isArray(list)) {
    return [];
  }
  return list.map((domain) => normalizeDomain(domain)).filter(Boolean);
}

async function saveDomainList(db, domains) {
  await db.run(
    `INSERT INTO site_settings (key, value, updated_at)
     VALUES ('domain_list', ?, CURRENT_TIMESTAMP)
     ON CONFLICT(key) DO UPDATE SET value = excluded.value, updated_at = CURRENT_TIMESTAMP`,
    [JSON.stringify(domains)]
  );
}

router.post("/login", async (req, res) => {
  try {
    const username = String(req.body?.username || "").trim();
    const password = String(req.body?.password || "");

    if (!username || !password) {
      return res.status(400).json({ error: "Username and password are required" });
    }

    const db = await getDb();
    const admin = await db.get("SELECT * FROM admin_users WHERE username = ?", [
      username
    ]);
    if (!admin) {
      return res.status(401).json({ error: "Invalid credentials" });
    }

    const passwordOk = await bcrypt.compare(password, admin.password_hash);
    if (!passwordOk) {
      return res.status(401).json({ error: "Invalid credentials" });
    }

    const token = signAdminToken({ id: admin.id, username: admin.username });
    setAdminCookie(res, token);

    return res.json({ ok: true, username: admin.username });
  } catch (error) {
    console.error("POST /admin/login error:", error);
    return res.status(500).json({ error: "Login failed" });
  }
});

router.post("/logout", requireAdmin, async (_req, res) => {
  clearAdminCookie(res);
  res.json({ ok: true });
});

router.get("/me", requireAdmin, (req, res) => {
  res.json({
    id: req.admin.id,
    username: req.admin.username
  });
});

router.get("/stats", requireAdmin, async (_req, res) => {
  try {
    const db = await getDb();
    const totals = await db.get(
      `SELECT
         (SELECT COUNT(*) FROM mail_sessions) AS total_sessions,
         (SELECT COUNT(*) FROM mail_sessions WHERE is_active = 1) AS active_sessions,
         (SELECT COUNT(*) FROM session_events) AS total_events,
         (SELECT COUNT(*) FROM mail_sessions WHERE date(created_at) = date('now')) AS sessions_today`
    );
    return res.json(totals);
  } catch (error) {
    console.error("GET /admin/stats error:", error);
    return res.status(500).json({ error: "Failed to load stats" });
  }
});

router.get("/sessions", requireAdmin, async (req, res) => {
  try {
    const limit = Math.min(200, Number(req.query.limit || 100));
    const db = await getDb();
    const rows = await db.all(
      `SELECT id, public_token, email_addr, is_active, last_ip, created_at, updated_at
       FROM mail_sessions
       ORDER BY id DESC
       LIMIT ?`,
      [limit]
    );
    return res.json({ sessions: rows });
  } catch (error) {
    console.error("GET /admin/sessions error:", error);
    return res.status(500).json({ error: "Failed to load sessions" });
  }
});

router.patch("/sessions/:id/status", requireAdmin, async (req, res) => {
  try {
    const id = Number(req.params.id);
    const isActive = req.body?.isActive ? 1 : 0;
    if (!Number.isInteger(id)) {
      return res.status(400).json({ error: "Invalid session id" });
    }
    const db = await getDb();
    await db.run(
      `UPDATE mail_sessions
       SET is_active = ?, updated_at = CURRENT_TIMESTAMP
       WHERE id = ?`,
      [isActive, id]
    );
    return res.json({ ok: true });
  } catch (error) {
    console.error("PATCH /admin/sessions/:id/status error:", error);
    return res.status(500).json({ error: "Failed to update session status" });
  }
});

router.get("/logs", requireAdmin, async (req, res) => {
  try {
    const limit = Math.min(300, Number(req.query.limit || 200));
    const db = await getDb();
    const rows = await db.all(
      `SELECT
         e.id,
         e.session_id,
         s.email_addr,
         e.event_type,
         e.payload,
         e.created_at
       FROM session_events e
       JOIN mail_sessions s ON s.id = e.session_id
       ORDER BY e.id DESC
       LIMIT ?`,
      [limit]
    );
    return res.json({
      logs: rows.map((row) => ({
        ...row,
        payload: safeJsonParse(row.payload, row.payload)
      }))
    });
  } catch (error) {
    console.error("GET /admin/logs error:", error);
    return res.status(500).json({ error: "Failed to load logs" });
  }
});

router.get("/settings", requireAdmin, async (_req, res) => {
  try {
    const db = await getDb();
    const rows = await db.all("SELECT key, value FROM site_settings");
    const settings = {};
    for (const row of rows) {
      settings[row.key] = row.value;
    }
    return res.json({ settings });
  } catch (error) {
    console.error("GET /admin/settings error:", error);
    return res.status(500).json({ error: "Failed to load settings" });
  }
});

router.put("/settings", requireAdmin, async (req, res) => {
  try {
    const db = await getDb();
    const updates = req.body?.settings;
    if (!updates || typeof updates !== "object") {
      return res.status(400).json({ error: "settings object is required" });
    }

    for (const [key, value] of Object.entries(updates)) {
      if (!SETTINGS_ALLOWLIST.has(key)) {
        continue;
      }
      await db.run(
        `INSERT INTO site_settings (key, value, updated_at)
         VALUES (?, ?, CURRENT_TIMESTAMP)
         ON CONFLICT(key) DO UPDATE SET value = excluded.value, updated_at = CURRENT_TIMESTAMP`,
        [key, String(value)]
      );
    }
    return res.json({ ok: true });
  } catch (error) {
    console.error("PUT /admin/settings error:", error);
    return res.status(500).json({ error: "Failed to update settings" });
  }
});

router.get("/domains", requireAdmin, async (_req, res) => {
  try {
    const db = await getDb();
    const domains = await getDomainList(db);
    return res.json({ domains });
  } catch (error) {
    console.error("GET /admin/domains error:", error);
    return res.status(500).json({ error: "Failed to load domains" });
  }
});

router.post("/domains", requireAdmin, async (req, res) => {
  try {
    const domain = normalizeDomain(req.body?.domain);
    if (!isValidDomain(domain)) {
      return res.status(400).json({ error: "Invalid domain format" });
    }

    const db = await getDb();
    const domains = await getDomainList(db);

    if (domains.includes(domain)) {
      return res.status(409).json({ error: "Domain already exists" });
    }

    domains.push(domain);
    await saveDomainList(db, domains);
    return res.json({ ok: true, domains });
  } catch (error) {
    console.error("POST /admin/domains error:", error);
    return res.status(500).json({ error: "Failed to add domain" });
  }
});

router.delete("/domains/:domain", requireAdmin, async (req, res) => {
  try {
    const domain = normalizeDomain(decodeURIComponent(req.params.domain || ""));
    const db = await getDb();
    const domains = await getDomainList(db);
    const filtered = domains.filter((item) => item !== domain);

    if (filtered.length === domains.length) {
      return res.status(404).json({ error: "Domain not found" });
    }

    await saveDomainList(db, filtered);
    return res.json({ ok: true, domains: filtered });
  } catch (error) {
    console.error("DELETE /admin/domains/:domain error:", error);
    return res.status(500).json({ error: "Failed to delete domain" });
  }
});

router.post("/change-password", requireAdmin, async (req, res) => {
  try {
    const currentPassword = String(req.body?.currentPassword || "");
    const newPassword = String(req.body?.newPassword || "");
    if (!currentPassword || !newPassword || newPassword.length < 8) {
      return res.status(400).json({
        error: "currentPassword and newPassword(>=8 chars) are required"
      });
    }

    const db = await getDb();
    const admin = await db.get("SELECT * FROM admin_users WHERE id = ?", [
      req.admin.id
    ]);
    if (!admin) {
      return res.status(404).json({ error: "Admin not found" });
    }

    const passwordOk = await bcrypt.compare(currentPassword, admin.password_hash);
    if (!passwordOk) {
      return res.status(401).json({ error: "Current password is incorrect" });
    }

    const hash = await bcrypt.hash(newPassword, 12);
    await db.run(
      "UPDATE admin_users SET password_hash = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?",
      [hash, admin.id]
    );
    return res.json({ ok: true });
  } catch (error) {
    console.error("POST /admin/change-password error:", error);
    return res.status(500).json({ error: "Failed to change password" });
  }
});

module.exports = router;
