const crypto = require("crypto");
const express = require("express");
const { getDb } = require("../db");
const guerrilla = require("../services/guerrilla");

const router = express.Router();

function makeToken() {
  return crypto.randomBytes(24).toString("hex");
}

function parseDomainList(rawValue) {
  try {
    const parsed = JSON.parse(rawValue || "[]");
    if (Array.isArray(parsed)) {
      return parsed.map((d) => String(d).trim()).filter(Boolean);
    }
  } catch (_error) {
    return [];
  }
  return [];
}

async function getSettings(db) {
  const rows = await db.all("SELECT key, value FROM site_settings");
  const map = {};
  for (const row of rows) {
    map[row.key] = row.value;
  }
  return map;
}

function pickDomain(requestedDomain, allowedDomains) {
  if (requestedDomain && allowedDomains.includes(requestedDomain)) {
    return requestedDomain;
  }
  return allowedDomains[0] || null;
}

async function getSessionByToken(db, token) {
  return db.get("SELECT * FROM mail_sessions WHERE public_token = ?", [token]);
}

async function logEvent(db, sessionId, eventType, payload = {}) {
  await db.run(
    "INSERT INTO session_events (session_id, event_type, payload) VALUES (?, ?, ?)",
    [sessionId, eventType, JSON.stringify(payload)]
  );
}

function getClientIp(req) {
  const xff = req.headers["x-forwarded-for"];
  if (typeof xff === "string" && xff.length > 0) {
    return xff.split(",")[0].trim();
  }
  return req.socket.remoteAddress || null;
}

router.get("/options", async (_req, res) => {
  try {
    const db = await getDb();
    const settings = await getSettings(db);
    res.json({
      siteName: settings.site_name || "Temp Mail Control",
      notice: settings.notice || "",
      domains: parseDomainList(settings.domain_list)
    });
  } catch (error) {
    console.error("GET /options error:", error);
    res.status(500).json({ error: "Failed to load options" });
  }
});

router.get("/session", async (req, res) => {
  try {
    const token = String(req.query.sessionToken || "");
    if (!token) {
      return res.status(400).json({ error: "sessionToken is required" });
    }
    const db = await getDb();
    const session = await getSessionByToken(db, token);
    if (!session) {
      return res.status(404).json({ error: "Session not found" });
    }
    if (!session.is_active) {
      return res.status(403).json({ error: "Session is disabled by admin" });
    }
    return res.json({
      sessionToken: session.public_token,
      emailAddress: session.email_addr
    });
  } catch (error) {
    console.error("GET /session error:", error);
    return res.status(500).json({ error: "Failed to load session" });
  }
});

router.post("/new", async (req, res) => {
  try {
    const db = await getDb();
    const settings = await getSettings(db);
    const allowedDomains = parseDomainList(settings.domain_list);
    if (!allowedDomains.length) {
      return res.status(400).json({
        error: "No domains configured. Ask admin to add domains first."
      });
    }
    const selectedDomain = pickDomain(req.body?.domain, allowedDomains);

    const mailbox = await guerrilla.createMailbox(selectedDomain);
    const token = makeToken();
    const ip = getClientIp(req);

    const result = await db.run(
      `INSERT INTO mail_sessions (public_token, sid_token, email_addr, last_ip)
       VALUES (?, ?, ?, ?)`,
      [token, mailbox.sidToken, mailbox.emailAddress, ip]
    );

    await logEvent(db, result.lastID, "mailbox_created", {
      email: mailbox.emailAddress,
      domain: selectedDomain
    });

    return res.json({
      sessionToken: token,
      emailAddress: mailbox.emailAddress
    });
  } catch (error) {
    console.error("POST /new error:", error);
    return res.status(500).json({ error: "Failed to create mailbox" });
  }
});

router.post("/regenerate", async (req, res) => {
  try {
    const sessionToken = String(req.body?.sessionToken || "");
    if (!sessionToken) {
      return res.status(400).json({ error: "sessionToken is required" });
    }

    const db = await getDb();
    const settings = await getSettings(db);
    const allowedDomains = parseDomainList(settings.domain_list);
    if (!allowedDomains.length) {
      return res.status(400).json({
        error: "No domains configured. Ask admin to add domains first."
      });
    }
    const selectedDomain = pickDomain(req.body?.domain, allowedDomains);

    const session = await getSessionByToken(db, sessionToken);
    if (!session) {
      return res.status(404).json({ error: "Session not found" });
    }
    if (!session.is_active) {
      return res.status(403).json({ error: "Session is disabled by admin" });
    }

    const mailbox = await guerrilla.regenerateMailbox(
      session.sid_token,
      selectedDomain
    );

    await db.run(
      `UPDATE mail_sessions
       SET email_addr = ?, updated_at = CURRENT_TIMESTAMP, last_ip = ?
       WHERE id = ?`,
      [mailbox.emailAddress, getClientIp(req), session.id]
    );
    await logEvent(db, session.id, "mailbox_regenerated", {
      email: mailbox.emailAddress,
      domain: selectedDomain
    });

    return res.json({
      sessionToken: session.public_token,
      emailAddress: mailbox.emailAddress
    });
  } catch (error) {
    console.error("POST /regenerate error:", error);
    return res.status(500).json({ error: "Failed to regenerate mailbox" });
  }
});

router.get("/messages", async (req, res) => {
  try {
    const sessionToken = String(req.query.sessionToken || "");
    if (!sessionToken) {
      return res.status(400).json({ error: "sessionToken is required" });
    }

    const db = await getDb();
    const session = await getSessionByToken(db, sessionToken);
    if (!session) {
      return res.status(404).json({ error: "Session not found" });
    }
    if (!session.is_active) {
      return res.status(403).json({ error: "Session is disabled by admin" });
    }

    const settings = await getSettings(db);
    const limit = Number(settings.messages_limit || "50");
    const messages = await guerrilla.listMessages(session.sid_token);

    await db.run(
      `UPDATE mail_sessions SET updated_at = CURRENT_TIMESTAMP, last_ip = ? WHERE id = ?`,
      [getClientIp(req), session.id]
    );
    await logEvent(db, session.id, "messages_refreshed", {
      count: messages.length
    });

    return res.json({
      count: messages.length,
      messages: messages.slice(0, Number.isFinite(limit) ? limit : 50)
    });
  } catch (error) {
    console.error("GET /messages error:", error);
    return res.status(500).json({ error: "Failed to fetch messages" });
  }
});

router.get("/messages/:id", async (req, res) => {
  try {
    const sessionToken = String(req.query.sessionToken || "");
    const mailId = String(req.params.id || "");
    if (!sessionToken) {
      return res.status(400).json({ error: "sessionToken is required" });
    }
    if (!mailId) {
      return res.status(400).json({ error: "id is required" });
    }

    const db = await getDb();
    const session = await getSessionByToken(db, sessionToken);
    if (!session) {
      return res.status(404).json({ error: "Session not found" });
    }
    if (!session.is_active) {
      return res.status(403).json({ error: "Session is disabled by admin" });
    }

    const message = await guerrilla.fetchMessage(session.sid_token, mailId);
    await logEvent(db, session.id, "message_opened", { id: mailId });
    return res.json(message);
  } catch (error) {
    console.error("GET /messages/:id error:", error);
    return res.status(500).json({ error: "Failed to fetch message" });
  }
});

router.delete("/messages/:id", async (req, res) => {
  try {
    const sessionToken = String(req.query.sessionToken || "");
    const mailId = String(req.params.id || "");
    if (!sessionToken) {
      return res.status(400).json({ error: "sessionToken is required" });
    }
    if (!mailId) {
      return res.status(400).json({ error: "id is required" });
    }

    const db = await getDb();
    const session = await getSessionByToken(db, sessionToken);
    if (!session) {
      return res.status(404).json({ error: "Session not found" });
    }
    if (!session.is_active) {
      return res.status(403).json({ error: "Session is disabled by admin" });
    }

    await guerrilla.deleteMessages(session.sid_token, [mailId]);
    await logEvent(db, session.id, "message_deleted", { id: mailId });
    return res.json({ ok: true });
  } catch (error) {
    console.error("DELETE /messages/:id error:", error);
    return res.status(500).json({ error: "Failed to delete message" });
  }
});

router.post("/messages/delete-all", async (req, res) => {
  try {
    const sessionToken = String(req.body?.sessionToken || "");
    if (!sessionToken) {
      return res.status(400).json({ error: "sessionToken is required" });
    }
    const db = await getDb();
    const session = await getSessionByToken(db, sessionToken);
    if (!session) {
      return res.status(404).json({ error: "Session not found" });
    }
    if (!session.is_active) {
      return res.status(403).json({ error: "Session is disabled by admin" });
    }

    const messages = await guerrilla.listMessages(session.sid_token);
    const ids = messages.map((m) => m.mail_id);
    await guerrilla.deleteMessages(session.sid_token, ids);
    await logEvent(db, session.id, "messages_deleted_all", { count: ids.length });

    return res.json({ ok: true, deleted: ids.length });
  } catch (error) {
    console.error("POST /messages/delete-all error:", error);
    return res.status(500).json({ error: "Failed to delete messages" });
  }
});

module.exports = router;
