const crypto = require("crypto");

const API_BASE =
  process.env.GUERRILLA_API_BASE || "https://api.guerrillamail.com/ajax.php";

async function fetchJson(url, options = {}, timeoutMs = 12000) {
  const controller = new AbortController();
  const timeout = setTimeout(() => controller.abort(), timeoutMs);
  try {
    const response = await fetch(url, { ...options, signal: controller.signal });
    if (!response.ok) {
      throw new Error(`HTTP ${response.status}`);
    }
    return await response.json();
  } finally {
    clearTimeout(timeout);
  }
}

function randomUserPrefix() {
  return crypto.randomBytes(5).toString("hex");
}

async function getSession() {
  const url = `${API_BASE}?${new URLSearchParams({ f: "get_email_address" }).toString()}`;
  const data = await fetchJson(url);
  if (!data.sid_token) {
    throw new Error("Failed to create GuerrillaMail session");
  }
  return data;
}

async function setEmailUser({ sidToken, userName, domain }) {
  const url = `${API_BASE}?f=set_email_user`;
  const body = new URLSearchParams({
    sid_token: sidToken,
    email_user: userName,
    domain
  });
  const data = await fetchJson(url, {
    method: "POST",
    headers: { "Content-Type": "application/x-www-form-urlencoded" },
    body: body.toString()
  });
  if (!data.email_addr) {
    throw new Error("Failed to set mailbox username/domain");
  }
  return data;
}

async function createMailbox(domain) {
  const session = await getSession();
  const userName = randomUserPrefix();
  const configured = await setEmailUser({
    sidToken: session.sid_token,
    userName,
    domain
  });

  return {
    sidToken: session.sid_token,
    emailAddress: configured.email_addr
  };
}

async function regenerateMailbox(sidToken, domain) {
  const userName = randomUserPrefix();
  const configured = await setEmailUser({ sidToken, userName, domain });
  return {
    sidToken,
    emailAddress: configured.email_addr
  };
}

async function listMessages(sidToken) {
  const query = new URLSearchParams({
    f: "get_email_list",
    offset: "0",
    sid_token: sidToken
  });
  const data = await fetchJson(`${API_BASE}?${query.toString()}`);
  return data.list || [];
}

async function fetchMessage(sidToken, mailId) {
  const query = new URLSearchParams({
    f: "fetch_email",
    email_id: String(mailId),
    sid_token: sidToken
  });
  return fetchJson(`${API_BASE}?${query.toString()}`);
}

async function deleteMessages(sidToken, ids) {
  if (!ids.length) {
    return { ok: true };
  }
  const query = new URLSearchParams({ f: "del_email", sid_token: sidToken });
  for (const id of ids) {
    query.append("email_ids[]", String(id));
  }
  return fetchJson(`${API_BASE}?${query.toString()}`);
}

module.exports = {
  createMailbox,
  regenerateMailbox,
  listMessages,
  fetchMessage,
  deleteMessages
};
